<?php

namespace Modules\Hubtel\Processor;

use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Modules\Gateway\Contracts\PaymentProcessorInterface;
use Modules\Gateway\Contracts\RequiresCallbackInterface;
use Modules\Gateway\Contracts\RequiresCancelInterface;
use Modules\Gateway\Entities\PaymentLog;
use Modules\Gateway\Services\GatewayHelper;
use Modules\Hubtel\Entities\Hubtel;
use Modules\Hubtel\Response\HubtelResponse;

class HubtelProcessor implements PaymentProcessorInterface, RequiresCallbackInterface, RequiresCancelInterface
{
    private $hubtel;

    private $helper;

    private $data;

    private $callbackUrl;

    private $successUrl;

    private $cancelUrl;

    private $payload;

    /**
     * Hubtel constructor
     *
     * @return void
     */
    public function __construct()
    {
        $this->helper = GatewayHelper::getInstance();
    }

    /**
     * Setup the initials value
     *
     * @return void
     */
    private function setupData()
    {
        $this->data = $this->helper->getPurchaseData($this->helper->getPaymentCode());
        $this->hubtel = Hubtel::firstWhere('alias', moduleConfig('hubtel.alias'))->data;
        $this->callbackUrl = route('gateway.webhook', ['gateway' => 'hubtel']);
        $this->successUrl = route('gateway.callback', withOldQueryIntegrity(['gateway' => 'hubtel']));
        $this->cancelUrl = route('gateway.cancel', withOldQueryIntegrity(['gateway' => 'hubtel']));
    }

    /**
     * Setup the payload values
     *
     * @return void
     */
    private function setPayload()
    {
        $this->payload = [
            'merchantId' => $this->hubtel->merchantId,
            'apiId' => $this->hubtel->apiId,
            'apiKey' => $this->hubtel->apiKey,
            'total' => $this->data->total,
            'clientReference' => $this->data->code,
            'callbackUrl' => $this->callbackUrl,
            'returnUrl' => $this->successUrl,
            'cancellationUrl' => $this->cancelUrl,
            'note' => $this->data->note,
            'cur' => $this->data->currency_code,
        ];
    }

    /**
     * Payment function
     *
     * @param  array|mix  $request
     * @return string $response
     */
    public function pay($request)
    {
        $this->setupData();
        $this->setPayload();
        if (strtoupper($this->data->currency_code) != 'GHS') {
            throw new \Exception(__('Currency not supported by merchant'));
        }
        $response = $this->callToApi($this->payload);

        return redirect($response['data']['checkoutUrl']);
    }

    /**
     * Verify payment transaction
     *
     * @param  array|mix  $request
     * @return array|mix $response
     */
    public function validateTransaction($request)
    {
        $this->setupData();

        return new HubtelResponse($this->data, $request);
    }

    /**
     * validate data from responses and make entry if data get validate
     */
    public function validatePayment(Request $request)
    {
        try {
            $webhookPayload = json_decode($request->getContent(), true);
            $clientReferenceCache = Cache::get($webhookPayload['Data']['ClientReference']);

            if (! empty($clientReferenceCache) && $webhookPayload['Data']['Status'] == 'Success') {
                $payment = $this->updatePaymentLog($webhookPayload, $clientReferenceCache);

                if ($payment) {
                    $this->updateOrderStatus($payment);
                    
                    $this->removeClientReference($clientReferenceCache);
                    
                    return true;
                    }

                return false;
            }
        } catch (\Exception $e) {
            \Log::error($e->getMessage());

            return false;
        }
    }

        /**
     * Update payment log based on webhook payload and client reference.
     *
     * @param  array  $webhookPayload The payload received from the webhook.
     * @param  mixed  $clientReferenceCache The client reference cached during payment initiation.
     * @return \Modules\Gateway\Entities\PaymentLog|null The updated payment log, or null if not found.
     */
    private function updatePaymentLog(array $webhookPayload, $clientReferenceCache): ?PaymentLog
    {
        $payment = PaymentLog::where('code', $clientReferenceCache)->first();

        if ($payment) {
            $payment->response_raw = $webhookPayload;
            $payment->status = 'completed';
            $payment->save();
        }

        return $payment;
    }

    /**
     * Update order status based on payment information.
     *
     * @param  \Modules\Gateway\Entities\PaymentLog  $payment The payment log.
     * @return void
     */
    private function updateOrderStatus(PaymentLog $payment): void
    {
        $order = Order::find($payment->code);

        if ($order) {
            $order->paid = $payment->total;
            $order->amount_received = $payment->total;
            $order->order_status_id = 3;
            $order->payment_status = 'Paid';
            $order->save();
        }
    }

    /**
     * Remove client reference from cache after processing the payment.
     *
     * @param  mixed  $clientReferenceCache The client reference cached during payment initiation.
     * @return void
     */
    private function removeClientReference($clientReferenceCache): void
    {
        Cache::forget($clientReferenceCache);
    }
    
    /**
     * Cancel Payment
     *
     * @param  object  $request
     * @return void
     */
    public function cancel($request)
    {
        throw new \Exception(__('Payment cancelled from Hubtel.'));
    }

    /**
     * Call API
     *
     * @param  bool  $setLocalhost
     * @param  array  $header
     * @return bool|string
     */
    public function callToApi($payload)
    {
        $url = 'https://payproxyapi.hubtel.com/items/initiate';
        $apiKey = $payload['apiKey'];
        $merchantId = $payload['merchantId'];
        $apiId = $payload['apiId'];
        $clientReference = 'HT'.time().$payload['clientReference'];
        $credentials = base64_encode($apiId . ':' . $apiKey);

        $authorization = 'Basic ' . $credentials;

        $data = [
            'totalAmount' => $payload['total'],
            'description' => $payload['note'] ?? 'Description',
            'callbackUrl' => $payload['callbackUrl'],
            'returnUrl' => $payload['returnUrl'],
            'merchantAccountNumber' => $merchantId,
            'cancellationUrl' => $payload['cancellationUrl'],
            'clientReference' => $clientReference,
        ];

        $curl = curl_init();

        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode([
                'totalAmount' => $data['totalAmount'],
                'description' => $data['description'],
                'callbackUrl' => $data['callbackUrl'],
                'returnUrl' => $data['returnUrl'],
                'merchantAccountNumber' => $data['merchantAccountNumber'],
                'cancellationUrl' => $data['cancellationUrl'],
                'clientReference' => $data['clientReference']
            ]),
            CURLOPT_HTTPHEADER => [
                "accept: application/json",
                "authorization: ".$authorization,
                "content-type: application/json"
            ],
        ]);

        $response = curl_exec($curl);
        curl_error($curl);
        curl_close($curl);
        $responseData = json_decode($response, true);

        if ($responseData['responseCode'] === '0000') {
            Cache::put($responseData['data']['clientReference'], $payload['clientReference']);

            return $responseData;
        }

        throw new \Exception(__('Failed to connect with Hubtel.'));
    }
}
