<?php

namespace Modules\AirtelAfrica\Processor;

use Modules\AirtelAfrica\Entities\AirtelAfrica;
use Modules\AirtelAfrica\Response\AirtelResponse;
use Modules\Gateway\Contracts\PaymentProcessorInterface;
use Modules\Gateway\Services\GatewayHelper;

class AirtelAfricaProcessor implements PaymentProcessorInterface
{
    private $airtel;

    private $response;

    private $baseUrl;

    private $accessToken;

    private $userInquiry;

    private $transaction;

    public function __construct()
    {
        $this->helper = GatewayHelper::getInstance();
        $this->airtel = AirtelAfrica::first()->data;
        $this->code = $this->helper->getPaymentCode();
        $this->purchaseData = $this->helper->getPurchaseData($this->code);
    }

    /**
     * clover pay
     *
     * @return AirtelResponse
     *
     * @throws \Exception
     */
    public function pay($request)
    {
        try {
            $this->setBaseUrl();
            $this->airtelAuthorizeToken();
            $this->userEnquiry($request->phone, $request->country);
            $response = $this->ussdPush($request->phone, $request->country);

            return new AirtelResponse($this->purchaseData, $response);

        } catch (\Exception $e) {
            paymentLog($e);

            throw new \Exception($e->getMessage());
        }

    }

    private function setBaseUrl()
    {
        if ($this->airtel->sandbox == '1') {
            $this->baseUrl = 'https://openapiuat.airtel.africa';
        } else {
            $this->baseUrl = 'https://openapi.airtel.africa';
        }
    }

    private function airtelAuthorizeToken()
    {
        $url = $this->baseUrl . '/auth/oauth2/token';
        $method = 'POST';
        $headers = [
            'Content-Type: application/json',
        ];
        $body = [
            'client_id' => $this->airtel->clientId,
            'client_secret' => $this->airtel->clientSecret,
            'grant_type' => 'client_credentials',
        ];
        $body = json_encode($body);

        try {
            $response = $this->curl($url, $method, $headers, $body);

            $response = json_decode($response, true);

            if (isset($response['access_token'])) {
                $this->accessToken = $response['access_token'];

                return;
            }

            throw new \Exception($response['error']);
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function userEnquiry($phone, $country)
    {
        $headers = [
            'Content-Type: application/json',
            'X-Country:' . $country,
            'X-Currency:' . $this->purchaseData->currency_code,
            'Authorization: Bearer ' . $this->accessToken,
        ];

        $url = $this->baseUrl . "/standard/v1/users/{$phone}";
        $method = 'GET';
        // Define array of request body.
        $body = [];

        try {
            $response = $this->curl($url, $method, $headers, $body);

            $response = json_decode($response, true);

            if (isset($response['status']) && $response['status']['success']) {
                return;
            }

            throw new \Exception(__('Invalid MSISDN'));
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function ussdPush($phone, $country)
    {
        $headers = [
            'Accept' => '*/* ',
            'Content-Type: application/json',
            'X-Country:' . $country,
            'X-Currency:' . $this->purchaseData->currency_code,
            'Authorization: Bearer ' . $this->accessToken,
        ];

        $uuid = uniqid();
        $this->transaction = $uuid;
        $body = [
            'reference'   => $uuid,
            'subscriber'  => [
                'country'  => $country,
                'currency' => $this->purchaseData->currency_code,
                'msisdn'   => $phone,
            ],
            'transaction' => [
                'amount'   => round($this->purchaseData->total, 0),
                'country'  => $country,
                'currency' => $this->purchaseData->currency_code,
                'id'       => $uuid,
            ],
        ];

        $body = json_encode($body);

        $url = $this->baseUrl . '/merchant/v1/payments/';
        $method = 'POST';

        try {
            $response = $this->curl($url, $method, $headers, $body);

            $result   = json_decode($response, true);

            if (isset($result['status'])) {
                $transactionQuery = $this->statusQuery($headers);

                if (isset($transactionQuery['status']) && $transactionQuery['status']['success']) {
                    return $transactionQuery;
                }
            }

            throw new \Exception(__('An error occurred while initiating the payment'));
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function statusQuery($headers)
    {
        $url = $this->baseUrl . "/standard/v1/payments/{$this->transaction}";
        $method = 'GET';
        $body = [];

        try {
            $response = $this->curl($url, $method, $headers, $body);
            $result   = json_decode($response, true);

            if (isset($result['status']) && $result['status']['success']) {
                return $result;
            }

            throw new \Exception(__('An error occurred while initiating the payment'));
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function curl($url, $method, $headers, $request_body)
    {
        $url = $url;
        $curl = curl_init();

        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_POSTFIELDS => $request_body,
            CURLOPT_HTTPHEADER => $headers,
        ]);

        $response = curl_exec($curl);

        curl_close($curl);

        return $response;
    }
}
