<?php

namespace Modules\DatabaseBackup\Http\Controllers;

use App\Lib\Env;
use Illuminate\Contracts\Support\Renderable;
use Modules\DatabaseBackup\DataTables\DatabaseBackupDataTable;
use Modules\DatabaseBackup\Http\Requests\DatabaseAutoBackupSettingRequest;
use App\Http\Controllers\Controller;
use Symfony\Component\Process\Process;
use Illuminate\Support\Facades\{
    Artisan,
    File,
    Log,
    Response,
    Storage
};

class DatabaseBackupController extends Controller
{
    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index()
    {
        return view('databasebackup::autobackup');
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function automatedDatabaseBackupForm()
    {
        $data['disks'] = moduleConfig('databasebackup.available_storage');
        return view('databasebackup::auto_backup_setting', $data);
    }

    /**
     * Store a newly created resource in storage.
     * @param DatabaseAutoBackupSettingRequest $request
     * @return Renderable
     */
    public function store(DatabaseAutoBackupSettingRequest $request)
    {
        try {
            Env::set('DATABASE_SCHEDULE_TYPE', $request->schedule_type);
            Env::set('IS_DATABASE_AUTOMATED_BACKUP', $request->is_database_automated_backup ? true : false);
            Env::set('DATABASE_STORAGE', $request->storage);
            
            $response = $this->messageArray(__('Auto backup setting setup successfully'), 'success');
            $this->setSessionValue($response);
            return to_route('database.automated.backup')->with($response);
        } catch (\Throwable $th) {

            $response = $this->messageArray(__('Auto backup setting setup Failed'), 'fail');
            return to_route('database.automated.backup')->with($response);
        }
    }

    /**
     * Show database backup list
     *
     * @param DatabaseBackupDataTable $dataTable
     * @return Renderable
     */
    public function list(DatabaseBackupDataTable $dataTable)
    {
        $directory = 'storage/' . config('backup.path');
        
        if (!Storage::exists($directory) && File::isWritable(storage_path())) {
            Storage::makeDirectory($directory);
        }

        return $dataTable->render('databasebackup::list');
    }

    /**
     * Download a database backup file.
     *
     * @param string $file
     * @return response
     */
    public function download($file)
    {
        try {
            $link = "storage/app/backup/database/$file";
            return Storage::download($link);
        } catch (\Exception $e) {

            $response = $this->messageArray(__('An error occurred while downloading the backup file.'), 'fail');
            $this->setSessionValue($response);
            return to_route('database.manual.backup.list');
        }
    }
    
    /**
     * Download a database backup file.
     *
     * @param string $file
     * @return response
     */
    public function destroy($file)
    {
        try {
            $link = "storage/app/backup/database/$file";
            
            Storage::delete($link);
            $response = $this->messageArray(__('Database backup file deleted successfully'), 'success');
            $this->setSessionValue($response);

            return to_route('database.manual.backup.list');
        } catch (\Exception $e) {

            $response = $this->messageArray(__('An error occurred while deleting the backup file.'), 'fail');
            $this->setSessionValue($response);
            return to_route('database.manual.backup.list');
        }
    }

    /**
     * Manually database backup
     *
     * @return redirect
     */
    public function manualDatabaseBackup()
    {
        try {
            $process = new Process(['mysqldump', '--version']);
            $process->run();
            
            if (!$process->isSuccessful()) {
                throw new \Exception(__("mysqldump is not available."));
            }
            
            $dbUserName = config('database.connections.mysql.username');
            $dbPassword = config('database.connections.mysql.password');
            $dbHost = config('database.connections.mysql.host');
            $dbDatabase = config('database.connections.mysql.database');
            
            $filename = date('Y-m-d-H-i-s') . ".sql";
            $path = storage_path("app/backup/database");
            
            if (!\Illuminate\Support\Facades\File::exists($path) && is_writable(storage_path())) {
                mkdir($path, 0777, true);
            }
            
            $command = "mysqldump --user=" .  $dbUserName . " --password=" . $dbPassword . " --host=" . $dbHost . " " .  $dbDatabase . " > " . $path . "/" . $filename;
            $returnVar = NULL;
            $output  = NULL;
            exec($command, $output, $returnVar);
            
            if ($returnVar !== 0) {
                throw new \Exception(__("An error occurred while executing the command. Return code") . ': ' . $returnVar);
            }
            
            return back()->withSuccess(__('Database backup successfully'));
        } catch (\Exception $e) {
            Log::error($e);
            return back()->withErrors($e->getMessage());
        }
    }
}
