<?php

namespace Modules\Leads\DataTables;

use App\DataTables\DataTable;
use Illuminate\Http\JsonResponse;
use Modules\Leads\Entities\Lead;

class LeadsDataTable extends DataTable
{
    /**
     * Handle the AJAX request for attribute groups.
     *
     * This function queries attribute groups and returns the data in a format suitable
     * for DataTables to consume via AJAX.
     */
    public function ajax(): JsonResponse
    {
        $leads = $this->query();

        $dt = datatables()
            ->of($leads)
            ->editColumn('first_name', function ($leads) {
                return ucfirst($leads->first_name . ' ' . $leads->last_name);
            })
            ->editColumn('company', function ($leads) {
                return ucfirst($leads->company);
            })
            ->editColumn('user_id', function ($leads) {
                return ! empty($leads->user) ? '<a href="' . route('users.edit', ['id' => $leads->user->id]) . '">' . wrapIt($leads->user->name, 10) . '</a>' : null;
            })
            ->editColumn('lead_status_id', function ($leads) {

                if (isset($leads->leadStatus)) {
                    return "<div class='badge badge-primary lead-status p-2' style='background-color:" . $leads->leadStatus->color . ";'><center><strong>" . $leads->leadStatus->name . '<strong></center></div>';
                }

                return null;
            })
            ->editColumn('lead_source_id', function ($leads) {
                return $leads->leadSource?->name;
            })
            ->editColumn('tags', function ($leads) {

                if (! empty($leads->tags)) {
                    $str2 = '';
                    $Html = "<div class='col-md-12'>";
                    $finalTag =  "<div class='tags-labels'>";
                    foreach ($leads->tags as $tagName) {
                        $str = '<span class="label label-tag p-2"><span class="tag-in-result">' . $tagName . '</span></span>';
                        if ($str2 == '') {
                            $str2 .= $str;
                        } else {
                            $str2 .= '<span class="hide"></span>' . $str;
                        }
                    }

                    $finalHtml = $Html . $finalTag . $str2 . '</div>' . '</div>';

                    return $finalHtml;
                }

                return null;
            });

        $dt->editColumn('created_at', function ($leads) {
            return $leads->format_created_at_only_date . '<br><span class="text-muted">' . $leads->format_created_at_only_time . '</span>';
        })
            ->addColumn('action', function ($leads) {

                $edit = '<a data-bs-toggle="tooltip" title="' . __('Edit') . '" href="' . route('admin.leads.edit', ['lead' => $leads->id]) . '" class="action-icon"><i class="feather icon-edit-1"></i></a>';

                $str = '';
                if ($this->hasPermission(['Modules\Leads\Http\Controllers\LeadsController@edit'])) {
                    $str .= $edit;
                }
                if ($this->hasPermission(['Modules\Leads\Http\Controllers\LeadsController@destroy'])) {
                    $str .= view('components.backend.datatable.delete-button', [
                        'route' => route('admin.leads.destroy', ['lead' => $leads->id]),
                        'id' => $leads->id,
                        'method' => 'DELETE',
                    ])->render();
                }

                return $str;
            })
            ->rawColumns(['first_name', 'company', 'user_id', 'lead_status_id', 'lead_source_id', 'tags', 'created_at', 'action']);

        return $dt->make(true);
    }

    /*
    * DataTable Query
    *
    * @return mixed
    */
    public function query()
    {
        $leads = Lead::select('leads.*')->filter();

        return $this->applyScopes($leads);
    }

    /*
    * DataTable HTML
    *
    * @return \Yajra\DataTables\Html\Builder
    */
    public function html()
    {
        $builder = $this->builder()
            ->addColumn(['data' => 'id', 'name' => 'id', 'title' => __('Id'), 'visible' => false])
            ->addColumn(['data' => 'first_name', 'name' => 'first_name', 'title' => __('Name'), 'className' => 'align-middle'])
            ->addColumn(['data' => 'company', 'name' => 'company', 'title' => __('Company'), 'className' => 'align-middle'])
            ->addColumn(['data' => 'user_id', 'name' => 'user_id', 'title' => __('Assigned'), 'className' => 'align-middle'])
            ->addColumn(['data' => 'lead_status_id', 'name' => 'lead_status_id', 'title' => __('Status'), 'className' => 'align-middle'])
            ->addColumn(['data' => 'lead_source_id', 'name' => 'lead_source_id', 'title' => __('Source'), 'className' => 'align-middle'])
            ->addColumn(['data' => 'tags', 'name' => 'tags', 'title' => __('Tags'), 'className' => 'align-middle']);

        $builder->addColumn(['data' => 'created_at', 'name' => 'created_at', 'title' => __('Created at'), 'className' => 'align-middle', 'width' => '11%'])
            ->addColumn([
                'data' => 'action', 'name' => 'action', 'title' => '', 'width' => '12%',
                'visible' => $this->hasPermission(['Modules\Leads\Http\Controllers\LeadsController@edit', 'Modules\Leads\Http\Controllers\LeadsController@destroy']),
                'orderable' => false, 'searchable' => false, 'className' => 'text-right align-middle',
            ])
            ->parameters(dataTableOptions(['dom' => 'Bfrtip']));

        return $builder;
    }

    public function setViewData()
    {
        $statusCounts = \DB::table('leads')
            ->join('lead_statuses', 'leads.lead_status_id', '=', 'lead_statuses.id')
            ->selectRaw('lead_statuses.name, COUNT(*) as count')
            ->groupBy('lead_statuses.name')
            ->pluck('count', 'lead_statuses.name');

        $this->data['groups'] = ['All' => $statusCounts->sum()] + $statusCounts->toArray();
    }
}
