<?php

/**
 * @author TechVillage <support@techvill.org>
 *
 * @contributor Md Shahnaouz Razu <[razu.techvill@gmail.com]>
 *
 * @created 19-08-24
 */

namespace Modules\PayMoney\Processor;

use Illuminate\Support\Facades\Session;
use Modules\Gateway\Contracts\PaymentProcessorInterface;
use Modules\Gateway\Contracts\RequiresCallbackInterface;
use Modules\Gateway\Facades\GatewayHelper;
use Modules\PayMoney\Entities\PayMoney;
use Modules\PayMoney\Response\PayMoneyResponse;
use PayMoney\Api\Amount;
use PayMoney\Api\Payer;
use PayMoney\Api\Payment;
use PayMoney\Api\RedirectUrls;
use PayMoney\Api\Transaction;

class PayMoneyProcessor implements PaymentProcessorInterface, RequiresCallbackInterface
{
    private $paymoney;

    private $data;

    private $code;

    private $name;

    /**
     * Setup method
     *
     * @return void
    */
    private function setup()
    {
        $this->paymoney = PayMoney::firstWhere('alias', moduleConfig('paymoney.alias'))->data;
        
        $this->code = GatewayHelper::getPaymentCode();
        
        $this->data = GatewayHelper::getPurchaseData($this->code);
        
        $this->name = moduleConfig('paymoney.name');
    }

    /**
     * Process payment using PayMoney gateway
     *
     * @param Request $request
     * @return RedirectResponse
     * @throws Exception
    */
    public function pay($request)
    {
        $this->setup();

        //Payer Object
        $payer = new Payer();
        $payer->setPaymentMethod($this->name);

        //Amount Object
        $amountIns = new Amount();
        $amountIns->setTotal(strval(round($this->data->total, 2)))->setCurrency($this->data->currency_code); //must give a valid currency code and must exist in merchant wallet list

        //Transaction Object
        $trans = new Transaction();
        $trans->setAmount($amountIns);

        //Store temporary integrity key in session
        Session::forget('integrity');
        Session::put('integrity', $request->integrity, now()->addMinutes(1));

        //RedirectUrls Object
        $urls = new RedirectUrls();
        $urls->setSuccessUrl(route(moduleConfig('gateway.payment_callback'), withOldQueryIntegrity(['gateway' => 'paymoney'])))
            ->setCancelUrl(route(moduleConfig('gateway.payment_cancel'), withOldQueryIntegrity(['gateway' => 'paymoney'])));

        //Payment Object
        $payment = new Payment();
        
        $payment->setCredentials([
            'client_id'     => $this->paymoney->client_id,
            'client_secret' => $this->paymoney->client_secret,
        ])
            ->setRedirectUrls($urls)
            ->setPayer($payer)
            ->setTransaction($trans);

        try {

            $payment->create();
            return redirect($payment->getApprovedUrl());

        } catch (\Exception $ex) {

            throw new \Exception($ex->getMessage());

        }

    }

    /**
     * Validate a PayMoney transaction
     *
     * This method verifies the integrity of a PayMoney transaction by decoding the response and checking its status.
     *
     * @param Request $request
     * @return PayMoneyResponse
     * @throws Exception If the transaction is invalid or an error occurs during validation
    */
    public function validateTransaction($request)
    {
        $this->setup();

        try {

            $payment_response = $request->all();

            $encoded_response = explode('?', $payment_response['integrity']);
            
            $transaction_data = json_decode(base64_decode(json_encode($encoded_response[1])), true);
            
            if ($transaction_data['status'] != 200) {
                throw new \Exception('Transaction failed');
            }

        } catch (\Throwable $th) {

            paymentLog($th);
            throw new \Exception($th->getMessage());

        }

        return new PayMoneyResponse($this->data, $transaction_data);
    }

    /**
     * Cancel a PayMoney payment
     *
     * This method cancels a PayMoney payment and throws an exception with a message indicating that the payment was cancelled.
     *
     * @throws \Exception Payment cancelled from paymoney.
    */
    public function cancel()
    {
        throw new \Exception(__('Payment cancelled from paymoney.'));
    }
}
