<?php

/**
 * @author TechVillage <support@techvill.org>
 *
 * @contributor Md Shahnaouz Razu <[razu.techvill@gmail.com]>
 *
 * @created 20-08-24
 */

namespace Modules\PayMoney\Response;

use Modules\Gateway\Contracts\HasDataResponseInterface;
use Modules\Gateway\Response\Response;

class PayMoneyResponse extends Response implements HasDataResponseInterface
{
    protected $response;

    private $data;

    /**
     * Constructor of the response
     *
     * @param object $data Order data object
     * @param object $response Payment response
     * @return $this
    */
    public function __construct($data, $response)
    {
        $this->data = $data;
        $this->response = $response;
        $this->updateStatus();

        return $this;
    }

    /**
     * Returns the raw payment response as a JSON string
     *
     * @return string JSON encoded payment response
    */
    public function getRawResponse(): string
    {
        return json_encode($this->response);
    }

    /**
     * Updates the payment status based on the response status code
     *
     * @return void
    */
    protected function updateStatus()
    {
        if ($this->response['status'] == 200) {
            $this->setPaymentStatus('completed');
        } else {
            $this->setPaymentStatus('failed');
        }
    }

    /**
     * Returns the payment response as a JSON string
     *
     * @return string JSON encoded payment response
    */
    public function getResponse(): string
    {
        return json_encode($this->getSimpleResponse());
    }

    /**
     * Returns a simplified payment response as an array
     *
     * @return array Simplified payment response data
    */
    private function getSimpleResponse()
    {
        return [
            'amount' => $this->data->total,
            'amount_captured' => $this->data->total,
            'currency' => $this->data->currency_code,
            'code' => $this->data->code,
        ];
    }

    /**
     * Returns the payment gateway name
     *
     * @return string The payment gateway name
    */
    public function getGateway(): string
    {
        return 'paymoney';
    }

    /**
     * Sets the payment status
     *
     * @param string $status The payment status to set
     * @return void
    */
    protected function setPaymentStatus($status)
    {
        $this->status = $status;
    }
}
