<?php

/**
 * @author TechVillage <support@techvill.org>
 *
 * @contributor Ashraful Rasel <[ashraful.techvill@gmail.com]>
 *
 * @created 11-11-2024
 */

namespace Modules\UddoktaPay\Processor;

use Illuminate\Support\Facades\Http;
use Modules\Gateway\Contracts\{
    PaymentProcessorInterface,
    RequiresCallbackInterface
};
use Modules\Gateway\Facades\GatewayHelper;
use Modules\UddoktaPay\Entities\UddoktaPay;
use Modules\UddoktaPay\Response\UddoktaPayResponse;

class UddoktaPayProcessor implements PaymentProcessorInterface, RequiresCallbackInterface
{
    private $code;

    private $apiKey;

    private $baseUrl;

    private $helper;

    private $purchaseData;

    private $successUrl;

    private $cancelUrl;

    private $requestBody;
    

    public function __construct()
    {
        $this->helper = GatewayHelper::getInstance();
    }

    /**
     * Handles payment for UddoktaPay
     *
     * @param \Illuminate\Http\Request
     * @return UddoktaPayResponse
     */
    public function pay($request)
    {
        $this->uddoktaPaySetup($request);

        $charge = $this->charge();

        if (! $charge || !isset($charge->payment_url) ) {
            throw new \Exception(__('Payment Request failed due to some issues. Please try again later.'));
        }

        return redirect($charge->payment_url);
    }

    /**
     * UddoktaPay data setup
     *
     * @param \Illuminate\Http\Request
     *
     * return mixed
     */
    private function uddoktaPaySetup($request)
    {
        try {
            
            $this->code = $this->helper->getPaymentCode();
            $this->purchaseData = $this->helper->getPurchaseData($this->code);

            $uddoktaPay = UddoktaPay::firstWhere('alias', 'uddoktapay')->data;

            $this->apiKey = $uddoktaPay->apiKey;

            $this->baseUrl = $uddoktaPay->baseUrl;


            $this->successUrl = route(moduleConfig('gateway.payment_callback'), withOldQueryIntegrity(['gateway' => 'uddoktapay']));
            $this->cancelUrl = route(moduleConfig('gateway.payment_cancel'), withOldQueryIntegrity(['gateway' => 'uddoktapay']));

            $this->requestBody = $this->getRequestBody();

        } catch (\Exception $e) {
            paymentLog($e);
            throw new \Exception(__('Error while trying to setup uddoktapay.'));
        }
    }

    /**
     * Create charge for payment
     *
     * @return mixed
     */
    private function charge()
    {
        try {

            if (strtoupper($this->purchaseData->currency_code) != 'BDT') {
                throw new \Exception(__('Currency not supported by merchant'));
            }

            $response = Http::withHeaders([
                    'RT-UDDOKTAPAY-API-KEY' => $this->apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ])
                ->post($this->baseUrl . '/api/checkout-v2', $this->requestBody);

            return json_decode($response->body()); 

        } catch (\Exception $e) {
            throw new \Exception($e->getMessage());
        }
    }


    private function getRequestBody()
    {
        return [
            'full_name'     => auth()->check() ?? auth()->name,
            'email'         => auth()->check() ?? auth()->email,
            'amount'        => number_format($this->purchaseData->total, 2),
            'metadata'      => [
                'code'   => $this->code,
            ],
            'redirect_url'  => $this->successUrl,
            'return_type'   => 'GET',
            'cancel_url'    => $this->cancelUrl,
            'webhook_url'   => $this->successUrl
        ];
    }



    public function validateTransaction($request)
    {
        try {
            $this->uddoktaPaySetup($request);

            if (!isset($request->invoice_id)) {
                throw new \Exception(__('Payment verification error with UddoktaPay'));
            }


            $originalIntegrity = $request->input('integrity');
            $modifiedintegrity = str_replace(' ', '+', $originalIntegrity);
            $request->merge(['integrity' => $modifiedintegrity]);

            $fields = [
                'invoice_id'     => $request->invoice_id
            ];

            $response = Http::withHeaders([
                    'RT-UDDOKTAPAY-API-KEY' => $this->apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ])
                ->post($this->baseUrl . '/api/verify-payment', $fields);

            $uddoktaPayResponse =  json_decode($response->body()); 

            return new UddoktaPayResponse($this->purchaseData,  $uddoktaPayResponse);

        } catch (\Exception $e) {
            throw new \Exception($e->getMessage());
        }

    }

    public function cancel($request)
    {
        throw new \Exception(__('Payment cancelled from UddoktaPay.'));
    }
}







